//
//  $Id: WXKWindowController.m 108 2009-06-24 14:54:10Z fujidana $
//  Copyright 2006 FUJIDANA. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKWindowController.h"
#import "WXKPasswordWindowController.h"
#import "WXKProgressWindowController.h"
#import "WXKError.h"


NSString *WXKToolbarReceiveItemIdentifier         = @"Receive Item Identifier";
NSString *WXKToolbarSendItemIdentifier            = @"Send Item Identifier";
NSString *WXKToolbarFetchListItemIdentifier       = @"Fetch List Item Identifier";
NSString *WXKToolbarAddressUtilityItemIdentifier  = @"Address Utility Item Identifier";
NSString *WXKToolbarBookmarkUtilityItemIdentifier = @"Bookmark Utility Item Identifier";
NSString *WXKToolbarFileUtilityItemIdentifier     = @"File Utility Item Identifier";
NSString *WXKToolbarMailUtilityItemIdentifier     = @"Mail Utility Item Identifier";

NSString *kWXKForceToAskPasswordKey               = @"kWXKForceToAskPasswordKey";


@interface WXKWindowController ()

- (void)communicateAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo;

@end


@implementation WXKWindowController

#pragma mark Class methods

+ (NSString *)metadataDirectory
{
	NSFileManager *fileManager = [NSFileManager defaultManager];
	NSArray       *paths       = NSSearchPathForDirectoriesInDomains(NSCachesDirectory, NSUserDomainMask, YES);
	if (paths == nil || [paths count] == 0) return NO;
	
	NSString *commonMetadataDirectory      = [[paths lastObject] stringByAppendingPathComponent:@"Metadata"];
	NSString *applicationMetadataDirectory = [commonMetadataDirectory stringByAppendingPathComponent:@"Kyopon Utilities"];
	
	if ([fileManager fileExistsAtPath:commonMetadataDirectory isDirectory:NULL] == NO)
	{
		[fileManager createDirectoryAtPath:commonMetadataDirectory attributes:nil];
	}
	if ([fileManager fileExistsAtPath:applicationMetadataDirectory isDirectory:NULL] == NO)
	{
		[fileManager createDirectoryAtPath:applicationMetadataDirectory attributes:nil];
	}
	
	return applicationMetadataDirectory;
}

#pragma mark override superclass

- (void)windowDidLoad
{
	[[self window] setExcludedFromWindowsMenu:YES];
	
	// disable to reorder by drag and drop
	[[self containerArrayController] setSupportsDragAndDrop:NO];
	[[self contentArrayController] setSupportsDragAndDrop:YES];
	[[self containerArrayController] setConditionallyHidesContainerColumnInContentTableView:YES];
	
	if (floor(NSAppKitVersionNumber) > 824) // if runtime is Leopard
	{
		NSTableView *parentTableView = [[self containerArrayController] tableView];
		[parentTableView setFocusRingType:NSFocusRingTypeNone];
		[parentTableView setUsesAlternatingRowBackgroundColors:NO];
		(void (*)(id, SEL, int))objc_msgSend(parentTableView, @selector(setSelectionHighlightStyle:), 1);
	}
	
	[super windowDidLoad];
}

#pragma mark Action methods and their didEndSelectors

- (IBAction)insertContent:(id)sender
{
	[[self contentArrayController] insert:sender];
}

- (IBAction)reindexContent:(id)sender
{
	if ([[self contentArrayController] respondsToSelector:@selector(reindexSelection:)])
	{
		[NSApp sendAction:@selector(reindexSelection:) to:[self contentArrayController] from:sender];
	}
}

- (IBAction)receive:(id)sender
{
	[[self managedObjectContext] commitEditing];
	[[self managedObjectContext] processPendingChanges];
	
	NSError *error;
	if ([[self managedObjectContext] save:&error] == NO)
	{
		[self presentError:error
			modalForWindow:[self window]
				  delegate:nil
		didPresentSelector:NULL
			   contextInfo:NULL];
		return;
	}
	
	if ([[[self contentArrayController] allObjects] count] > 0)
	{
		NSAlert *alert = [[[NSAlert alloc] init] autorelease];
		[alert addButtonWithTitle:NSLocalizedString(@"Receive", @"receiveAlert.receiveButton")];
		[alert addButtonWithTitle:NSLocalizedString(@"Cancel", @"alert.cancelButton")];
		[alert setMessageText:[NSString stringWithFormat:NSLocalizedString(@"Are you sure you want to overwrite every %@ on this computer with the data received from your Kyopon?", @"receiveAlert.message"), [[self contentArrayController] localizedEntityName]]];
		[alert setInformativeText:NSLocalizedString(@"Any changes you have made on this computer will be lost.", @"receiveAlert.information")];
		[alert setAlertStyle:NSInformationalAlertStyle];
		// [alert setHelpAnchor:@"ahkabhlp22"];
		// [alert setShowsHelp:YES];
		
		[alert beginSheetModalForWindow:[self window]
						  modalDelegate:self
						 didEndSelector:@selector(communicateAlertDidEnd:returnCode:contextInfo:)
							contextInfo:@selector(receiveWithPassword:progressController:error:)];
	}
	else
	{
		[self communicateAlertDidEnd:nil returnCode:NSAlertFirstButtonReturn contextInfo:@selector(receiveWithPassword:progressController:error:)];
	}
}

- (IBAction)send:(id)sender
{
	[[self managedObjectContext] commitEditing];
	[[self managedObjectContext] processPendingChanges];
	
	// Before sending, save the managed object context in order to check validation errors.
	NSError *error;
	if ([[self managedObjectContext] save:&error] == NO)
	{
		[self presentError:error
			modalForWindow:[self window]
				  delegate:nil
		didPresentSelector:NULL
			   contextInfo:NULL];
		return;
	}
	
	NSAlert *alert = [[[NSAlert alloc] init] autorelease];
	[alert addButtonWithTitle:NSLocalizedString(@"Send", @"sendAlert.sendButton")];
	[alert addButtonWithTitle:NSLocalizedString(@"Cancel", @"alert.cancelButton")];
	[alert setMessageText:[NSString stringWithFormat:NSLocalizedString(@"Are you sure you want to overwrite the %@ on your Kyopon with the data send from this computer?", @"sendSheet.message"), [[self contentArrayController] localizedEntityName]]];
	[alert setInformativeText:NSLocalizedString(@"Any changes you have made on your Kyopon will be lost.", @"sendAlert.information")];
	[alert setAlertStyle:NSWarningAlertStyle];
	// [alert setHelpAnchor:@"ahkabhlp21"];
	// [alert setShowsHelp:YES];
	
	[alert beginSheetModalForWindow:[self window]
					  modalDelegate:self
					 didEndSelector:@selector(communicateAlertDidEnd:returnCode:contextInfo:)
						contextInfo:@selector(sendWithPassword:progressController:error:)];
}

- (IBAction)fetchList:(id)sender
{
	[self communicateAlertDidEnd:nil returnCode:NSAlertFirstButtonReturn contextInfo:@selector(fetchListWithPassword:progressController:error:)];
}

/* contextInfo must not be NULL, but be a selector whose signature is as following:
 - (BOOL)communicateWithPassword:(NSString *)password progressController:(WXKProgressWindowController *)progress error:(NSError **)errorPtr;
 */
- (void)communicateAlertDidEnd:(NSAlert *)alert returnCode:(int)returnCode contextInfo:(void *)contextInfo
{
	if (returnCode == NSAlertFirstButtonReturn)
	{
		[[alert window] orderOut:nil];
		
		NSString *password = [[WXKPasswordWindowController sharedWindowController] password];
		
		// If password is obtained, begin to communicate with Kyopon.
		if (password != nil)
		{
			BOOL succeeded;
			NSError *error;
			
			WXKProgressWindowController *progress = [WXKProgressWindowController sharedWindowController];
			
			// display progress window
			[progress beginProgressWithWindow:[self window] message:NSLocalizedString(@"Prepapring for Communication...", @"progress.beginMessage")];
			
			// Actual communication is done here. Subclasses must implement a method designated 
			// in the contextInfo (selector). Typical selector names are: 
			// @selector(sendWithPassword:progressController:error:), 
			// @selector(receiveWithPassword:progressController:error:),
			// @selector(fetchListWithPassword:progressController:error:)
			succeeded = ((BOOL (*)(id, SEL, id, id, id *))objc_msgSend)(self, (SEL)contextInfo, password, progress, &error);
			
			// hide progress window
			[progress endProgress];
			
			// display error dialog 
			if (succeeded == NO)
			{
				[self presentError:error
					modalForWindow:[self window]
						  delegate:self
				didPresentSelector:@selector(communicationDidPresentErrorWithRecovery:contextInfo:)
					   contextInfo:contextInfo];
			}
		}
	}
}

- (void)communicationDidPresentErrorWithRecovery:(BOOL)didRecover contextInfo:(void *)contextInfo
{
	if (didRecover)
	{
		[self communicateAlertDidEnd:nil returnCode:NSAlertFirstButtonReturn contextInfo:contextInfo];
	}
}

#pragma mark NSErrorRecoveryAttempting informal protocol

- (void)attemptRecoveryFromError:(NSError *)error optionIndex:(unsigned int)recoveryOptionIndex delegate:(id)delegate didRecoverSelector:(SEL)didRecoverSelector contextInfo:(void *)contextInfo
{
	BOOL success = NO;
	
	if (recoveryOptionIndex == 0) // recovery requested 
	{
		// In case error reason is password...
		if ([error code] == WXKPhoneIncorrectPasswordError || 
			[error code] == WXKPhoneInapplicablePasswordStringLengthError ||
			[error code] == WXKPhoneInapplicablePasswordStringEncodingError)
		{
			// set to ask user input for next communication, 
			[[WXKPasswordWindowController sharedWindowController] setRequestsUserInput:YES];
		}
		success = YES;
	}
	
	// invoke didPresentSelector, which is designated in 
	// presentError:modalForWindow:delegate:didPresentSelector:contextInfo:.
	// The signature of didPresentSelector is like the following:
	// - (void)didPresentErrorWithRecovery:(BOOL)didRecover contextInfo:(void  *)contextInfo;
	if (delegate && didRecoverSelector)
	{
		NSMethodSignature *signature = [delegate methodSignatureForSelector:didRecoverSelector];
		NSInvocation *invoke = [NSInvocation invocationWithMethodSignature:signature];
		[invoke setSelector:didRecoverSelector];
		[invoke setArgument:(void *)&success atIndex:2];
		if (contextInfo) [invoke setArgument:&contextInfo atIndex:3];
		[invoke invokeWithTarget:delegate];
	}
}

#pragma mark override superclass (NSResponder)

- (NSError *)willPresentError:(NSError *)error
{
	if ([[error domain] isEqualToString:WXKErrorDomain])
	{
		if ([error code] == WXKPhoneUserCancelledError)
		{
		}
		else
		{
			NSMutableDictionary *newUserInfo = [NSMutableDictionary dictionary];
			if ([error userInfo] != nil)
			{
				[newUserInfo setDictionary:[error userInfo]];
			}
			
			NSString *recoverySuggestion;
			if ([error code] == WXKPhoneIncorrectPasswordError || 
				[error code] == WXKPhoneInapplicablePasswordStringLengthError ||
				[error code] == WXKPhoneInapplicablePasswordStringEncodingError)
			{
				// set recovery attempter, 
				recoverySuggestion = NSLocalizedString(@"Enter security code and try again?", @"incorrectPasswordError.recoverySuggestion");
			}
			else
			{
				recoverySuggestion = NSLocalizedString(@"Make sure that the cable is connected and that your cellphone is on stanby with its top screen displayed. Try again?", @"otherError.recoverySuggestion");
			}
			
			[newUserInfo setObject:recoverySuggestion forKey:NSLocalizedRecoverySuggestionErrorKey];
			[newUserInfo setObject:self forKey:NSRecoveryAttempterErrorKey];
			[newUserInfo setObject:[NSArray arrayWithObjects:NSLocalizedString(@"Try Again", @""), NSLocalizedString(@"Cancel", @""), nil] forKey:NSLocalizedRecoveryOptionsErrorKey];
			return [[error class] errorWithDomain:[error domain]
											 code:[error code] 
										 userInfo:newUserInfo];
		}
	}
	return [super willPresentError:error];
}

#pragma mark NSToolbar delegate

- (NSToolbarItem *)toolbar:(NSToolbar *)toolbar itemForItemIdentifier:(NSString *)itemIdentifier willBeInsertedIntoToolbar:(BOOL)flag
{
	NSToolbarItem *toolbarItem;
	
	if ([itemIdentifier isEqual:WXKToolbarReceiveItemIdentifier])
	{
		// Receive
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Receive", @"receiveToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Receive from Kyopon", @"receiveToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Receive data from Kyopon.", @"receiveToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarReceiveIcon.tiff"]];
		[toolbarItem setTarget:self];
		[toolbarItem setAction:@selector(receive:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarSendItemIdentifier])
	{
		// Send
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Send", @"sendToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Send to Kyopon", @"sendToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Send data to Kyopon.", @"sendToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarSendIcon.tiff"]];
		[toolbarItem setTarget:self];
		[toolbarItem setAction:@selector(send:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarFetchListItemIdentifier])
	{
		// Fetch list
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Fetch List", @"fetchListToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Fetch List", @"fetchListToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Fetch List from Kyopon.", @"fetchListToolbarItem.toolTip")];
		//		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarNewIcon.tiff"]];
		[toolbarItem setTarget:self];
		[toolbarItem setAction:@selector(fetchList:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarAddressUtilityItemIdentifier])
	{
		// Launch Kyopon Addresses
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Kyopon Addresses", @"kyoponAddressesToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Launch Kyopon Addresses", @"kyoponAddressesToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Launch Kyopon Addresses.", @"kyoponAddressesToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarAddressUtilityIcon.tiff"]];
		[toolbarItem setTarget:nil];
		[toolbarItem setAction:@selector(launchAddressUtility:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarBookmarkUtilityItemIdentifier])
	{
		// Launch Kyopon Bookmarks
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Kyopon Bookmarks", @"kyoponBookmarksToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Launch Kyopon Bookmarks", @"kyoponBookmarksToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Launch Kyopon Bookmarks.", @"kyoponBookmarksToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarBookmarkUtilityIcon.tiff"]];
		[toolbarItem setTarget:nil];
		[toolbarItem setAction:@selector(launchBookmarkUtility:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarFileUtilityItemIdentifier])
	{
		// Launch Kyopon Files
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Kyopon Files", @"kyoponFilesToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Launch Kyopon Files", @"kyoponFilesToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Launch Kyopon Files.", @"kyoponFilesToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarFileUtilityIcon.tiff"]];
		[toolbarItem setTarget:nil];
		[toolbarItem setAction:@selector(launchFileUtility:)];
	}
	else if ([itemIdentifier isEqual:WXKToolbarMailUtilityItemIdentifier])
	{
		// Launch Kyopon Mails
		toolbarItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
		[toolbarItem setLabel:NSLocalizedString(@"Kyopon Mails", @"kyoponMailsToolbarItem.label")];
		[toolbarItem setPaletteLabel:NSLocalizedString(@"Launch Kyopon Mails", @"kyoponMailsToolbarItem.paletteLabel")];
		[toolbarItem setToolTip:NSLocalizedString(@"Launch Kyopon Mails.", @"kyoponMailsToolbarItem.toolTip")];
		[toolbarItem setImage:[NSImage imageNamed:@"ToolbarMailUtilityIcon.tiff"]];
		[toolbarItem setTarget:nil];
		[toolbarItem setAction:@selector(launchMailUtility:)];
	}
	else
	{
		toolbarItem = [super toolbar:toolbar itemForItemIdentifier:itemIdentifier willBeInsertedIntoToolbar:flag];
	}
	
	return toolbarItem;
}

#pragma mark methods implementing NSValidatedUserInterfaceItem protocol

- (BOOL)validateUserInterfaceItem:(id <NSValidatedUserInterfaceItem>)theItem
{
	if ([theItem action] == @selector(showWindow:) && [(NSObject *)theItem isKindOfClass:[NSMenuItem class]])
	{
		[(NSMenuItem *)theItem setState:([[self window] isKeyWindow] ? NSOnState : NSOffState)];
		return YES;
	}
	if ([theItem action] == @selector(insertContent:) || [theItem action] == @selector(reindexContent:))
	{
		return [[self contentArrayController] validateUserInterfaceItem:theItem];
	}
	return [super validateUserInterfaceItem:theItem];
}

@end


#pragma mark -


@implementation FJNContainerArrayController (WXKExtensions)

- (NSArray *)initializedObjectsWithDictionaries:(NSArray *)dictionaries
{
	NSManagedObjectContext *context      = [self managedObjectContext];
	NSFetchRequest         *fetchRequest = [[[NSFetchRequest alloc] init] autorelease];
	NSEntityDescription    *entity       = [NSEntityDescription entityForName:[self entityName] inManagedObjectContext:context];
	[fetchRequest setEntity:entity];
	[fetchRequest setSortDescriptors:[self defaultSortDescriptors]];
	
	NSError *fetchError;
	NSArray *fetchResults = [context executeFetchRequest:fetchRequest error:&fetchError];
	
	NSManagedObject *object;
	NSDictionary *keyedValues;
	unsigned count = [dictionaries count];
	
	if (fetchResults == nil)
	{
		// If fetch fails, show the error and terminate the application.
		[NSApp presentError:fetchError];
		[NSApp terminate:nil];
	}
	else if ([fetchResults count] == count)
	{
		[context processPendingChanges];
		[[context undoManager] disableUndoRegistration];
		
		int i;
		for (i = 0; i < count; i++)
		{
			object = [fetchResults objectAtIndex:i];
			keyedValues = [dictionaries objectAtIndex:i];
			[object setValuesForKeysWithDictionary:keyedValues];
		}
		[context processPendingChanges];
		[[context undoManager] enableUndoRegistration];
		
		return fetchResults;
	}
	else
	{
		[context processPendingChanges];
		[[context undoManager] disableUndoRegistration];
		
		// delete all existing objects
		NSEnumerator *enumerator = [fetchResults objectEnumerator];
		while (object = [enumerator nextObject])
		{
			[context deleteObject:object];
		}
		
		// create objects and set properties
		NSMutableArray *array = [NSMutableArray arrayWithCapacity:count];
		int i;
		for (i = 0; i < count; i++)
		{
			object = [NSEntityDescription insertNewObjectForEntityForName:[self entityName] inManagedObjectContext:context];
			keyedValues = [dictionaries objectAtIndex:i];
			[object setValuesForKeysWithDictionary:keyedValues];
			[array addObject:object];
		}
		
		[context processPendingChanges];
		[[context undoManager] enableUndoRegistration];
		
		return array;
	}
	return nil;
}

@end
